function New-AWSBeanstalkApp {
	[CmdletBinding()]
	param (
		[Parameter(Mandatory)]
		[ValidateNotNullOrEmpty()]
		[string]$ApplicationName,

		[Parameter(Mandatory)]
		[ValidateNotNullOrEmpty()]
		[string]$Description,

		[Parameter(Mandatory)]
		[ValidateNotNullOrEmpty()]
		[string]$EnvironmentType,

		[Parameter(Mandatory)]
		[ValidateNotNullOrEmpty()]
		[string[]]$CreateEnvironmentName,

		[Parameter(Mandatory)]
		[ValidateNotNullOrEmpty()]
		[string]$DeployToEnvironmentName,

		[Parameter(Mandatory)]
		[ValidateNotNullOrEmpty()]
		[string]$PackageFolderPath,

		[Parameter(Mandatory)]
		[ValidateNotNullOrEmpty()]
		[string]$S3BucketName,

		[Parameter()]
		[ValidateNotNullOrEmpty()]
		[int]$WaitForEnvironment = 15 ## minuty
	)

	try {
		if (-not ($ebApp = Get-EBApplication -ApplicationName $ApplicationName)) {
			$ebApp = New-EBApplication -ApplicationName $ApplicationName -Description $Description
		} else {
			Write-Verbose -Message "Aplikacja BeanStalk [$($ApplicationName)] już istnieje."
		}
		
		$params = @{
			ApplicationName = $ApplicationName
		}
		$CreateEnvironmentName | Where-Object { $_ -notin (Get-EBEnvironment -ApplicationName $ApplicationName | Select-Object -ExpandProperty EnvironmentName) } | ForEach-Object {
			$null = New-EBEnvironment @params -EnvironmentName $_ -SolutionStackName $EnvironmentType -Tier_Type Standard -Tier_Name WebServer
		}
		
		$stopwatch = [system.diagnostics.stopwatch]::StartNew()
		if (@(Get-EBEnvironment -ApplicationName $ApplicationName -EnvironmentName $CreateEnvironmentName).where({ $_.Health -ne 'Green'})) {
			while (($stopwatch.Elapsed.TotalMinutes -lt $WaitForEnvironment) -and (Get-EBEnvironment -ApplicationName $ApplicationName -EnvironmentName $CreateEnvironmentName).where({ $_.Health -ne 'Green'})) {
				Write-Verbose 'Poczekaj na utworzenie środowiska...'
				Start-Sleep -Seconds 60
			}
			if ($stopWatch.Elapsed.TotalMinutes -gt $WaitForEnvironment) {
				Write-Error -Message 'Upłynął limit czasu procesu tworzenia środowiska.'
			} else {
				Write-Verbose -Message 'Środowisko zostało utworzone!'
			}
			$stopwatch.Stop()
		} else {
			Write-Verbose -Message 'Wszystkie środowiska już zostały utworzone.'
		}

		## Pakowanie foldera pakietu do pliku zip 
		$pkgZipPath = "$env:TEMP\$ApplicationName.zip"
		Compress-Archive -Path "$PackageFolderPath\*" -DestinationPath $pkgZipPath

		## Przesyłanie pakietu do S3
		Write-S3Object -BucketName $S3BucketName -File $pkgZipPath

		## Tworzenie nowej wersji 
		$verLabel = [System.DateTime]::Now.Ticks.ToString()
		$newVerParams = @{
			ApplicationName       = $ApplicationName
			VersionLabel          = $verLabel
			SourceBundle_S3Bucket = $S3BucketName
			SourceBundle_S3Key    = "$ApplicationName.zip"
		}
		$null = New-EBApplicationVersion @newVerParams

		$null = Update-EBEnvironment -ApplicationName $ApplicationName -EnvironmentName $DeployToEnvironmentName -VersionLabel $verLabel -Force
		$stopwatch = [system.diagnostics.stopwatch]::StartNew()
		while (($stopwatch.Elapsed.TotalMinutes -lt $WaitForEnvironment) -and (Get-EBEnvironment -ApplicationName $ApplicationName -EnvironmentName $DeployToEnvironmentName).where({ $_.Health -ne 'Green'})) {
			Write-Verbose 'Poczekaj na zaktualizowanie środowiska...'
			Start-Sleep -Seconds 60
		}
		if ($stopWatch.Elapsed.TotalMinutes -gt $WaitForEnvironment) {
			Write-Error -Message 'Aktualizacja środowiska się nie powiodła.'
		} else {
			Write-Verbose -Message 'Aktualizacja środowiska zakończyła się pomyślnie!'
		}
		$stopwatch.Stop()

	} catch {
		$PSCmdlet.ThrowTerminatingError($_)
	} finally {
		if (Get-Variable -Name pkgZipPath -ErrorAction Ignore) {
			Remove-Item -Path $pkgZipPath -Force -ErrorAction Ignore
		}
	}
}